// Perch Binary Codec - VarInt encoding and message serialization
// Compatible with perch/src/web/webtransport/codec.rs

// Wire message types (must match Rust codec)
export const Wire = {
  // Extension -> Server (datagrams)
  LOG: 0x02,
  NETWORK: 0x03,
  URL_UPDATE: 0x04,
  PONG: 0x05,

  // Control stream (bidi)
  REGISTER: 0x01,
  REGISTER_OK: 0x06,
  REGISTER_ERROR: 0x07,

  // Extension -> Server (uni stream)
  SCREENSHOT_DATA: 0x10,

  // Server -> Extension (commands via bidi)
  PING: 0x20,
  REQUEST_SCREENSHOT: 0x21,
  EXECUTE_JS: 0x22,
  CLICK: 0x23,
  FILL: 0x24,
  GET_TEXT: 0x25,
  NAVIGATE: 0x26,

  // Extension -> Server (command response)
  COMMAND_RESULT: 0x30,
};

// VarInt encoding (QUIC-style)
export class VarInt {
  static encode(value, writer) {
    const v = BigInt(value);
    if (v < (1n << 6n)) {
      writer.writeUint8(Number(v));
    } else if (v < (1n << 14n)) {
      writer.writeUint16((0b01 << 14) | Number(v));
    } else if (v < (1n << 30n)) {
      writer.writeUint32((0b10 << 30) | Number(v));
    } else {
      writer.writeUint64((0b11n << 62n) | v);
    }
  }

  static async decode(reader) {
    const b = await reader.readUint8();
    const tag = b >> 6;

    switch (tag) {
      case 0b00:
        return BigInt(b & 0x3f);
      case 0b01: {
        const b2 = await reader.readUint8();
        return BigInt(((b & 0x3f) << 8) | b2);
      }
      case 0b10: {
        const rest = await reader.readBytes(3);
        return BigInt(
          ((b & 0x3f) << 24) | (rest[0] << 16) | (rest[1] << 8) | rest[2]
        );
      }
      case 0b11: {
        const rest = await reader.readBytes(7);
        let v = BigInt(b & 0x3f);
        for (const byte of rest) {
          v = (v << 8n) | BigInt(byte);
        }
        return v;
      }
    }
  }
}

// String encoding (varint length + UTF-8 bytes)
export class StringCodec {
  static encode(str, writer) {
    const bytes = new TextEncoder().encode(str);
    VarInt.encode(bytes.length, writer);
    writer.writeBytes(bytes);
  }

  static async decode(reader) {
    const len = await VarInt.decode(reader);
    const bytes = await reader.readBytes(Number(len));
    return new TextDecoder().decode(bytes);
  }
}

// Bytes encoding (varint length + raw bytes)
export class BytesCodec {
  static encode(bytes, writer) {
    VarInt.encode(bytes.length, writer);
    writer.writeBytes(bytes);
  }

  static async decode(reader) {
    const len = await VarInt.decode(reader);
    return await reader.readBytes(Number(len));
  }
}

// Helper for writing bytes
export class ByteWriter {
  constructor() {
    this.chunks = [];
    this.size = 0;
  }

  writeUint8(v) {
    this.chunks.push(new Uint8Array([v]));
    this.size += 1;
  }

  writeUint16(v) {
    const arr = new Uint8Array(2);
    arr[0] = (v >> 8) & 0xff;
    arr[1] = v & 0xff;
    this.chunks.push(arr);
    this.size += 2;
  }

  writeUint32(v) {
    const arr = new Uint8Array(4);
    arr[0] = (v >> 24) & 0xff;
    arr[1] = (v >> 16) & 0xff;
    arr[2] = (v >> 8) & 0xff;
    arr[3] = v & 0xff;
    this.chunks.push(arr);
    this.size += 4;
  }

  writeUint64(v) {
    const arr = new Uint8Array(8);
    let bigV = BigInt(v);
    for (let i = 7; i >= 0; i--) {
      arr[i] = Number(bigV & 0xffn);
      bigV = bigV >> 8n;
    }
    this.chunks.push(arr);
    this.size += 8;
  }

  writeBytes(bytes) {
    this.chunks.push(bytes instanceof Uint8Array ? bytes : new Uint8Array(bytes));
    this.size += bytes.length;
  }

  toBytes() {
    const result = new Uint8Array(this.size);
    let offset = 0;
    for (const chunk of this.chunks) {
      result.set(chunk, offset);
      offset += chunk.length;
    }
    return result;
  }
}

// Helper for reading bytes from a stream
export class StreamReader {
  constructor(reader) {
    this.reader = reader;
    this.buffer = new Uint8Array(0);
    this.offset = 0;
  }

  async ensureBytes(n) {
    while (this.buffer.length - this.offset < n) {
      const { value, done } = await this.reader.read();
      if (done) throw new Error('Stream ended unexpectedly');
      const newBuffer = new Uint8Array(this.buffer.length - this.offset + value.length);
      newBuffer.set(this.buffer.subarray(this.offset));
      newBuffer.set(value, this.buffer.length - this.offset);
      this.buffer = newBuffer;
      this.offset = 0;
    }
  }

  async readUint8() {
    await this.ensureBytes(1);
    return this.buffer[this.offset++];
  }

  async readUint16() {
    await this.ensureBytes(2);
    const v = (this.buffer[this.offset] << 8) | this.buffer[this.offset + 1];
    this.offset += 2;
    return v;
  }

  async readUint32() {
    await this.ensureBytes(4);
    const v =
      (this.buffer[this.offset] << 24) |
      (this.buffer[this.offset + 1] << 16) |
      (this.buffer[this.offset + 2] << 8) |
      this.buffer[this.offset + 3];
    this.offset += 4;
    return v >>> 0; // Convert to unsigned
  }

  async readBytes(n) {
    await this.ensureBytes(n);
    const result = this.buffer.slice(this.offset, this.offset + n);
    this.offset += n;
    return result;
  }
}

// Encode a log datagram
export function encodeLogDatagram(level, message, url, timestamp, sourceFile, lineNumber) {
  const writer = new ByteWriter();
  writer.writeUint8(Wire.LOG);
  writer.writeUint8(level);
  StringCodec.encode(message, writer);
  StringCodec.encode(url, writer);
  VarInt.encode(timestamp, writer);
  StringCodec.encode(sourceFile || '', writer);
  VarInt.encode(lineNumber || 0, writer);
  return writer.toBytes();
}

// Encode a network datagram
export function encodeNetworkDatagram(method, url, status, durationMs, timestamp) {
  const writer = new ByteWriter();
  writer.writeUint8(Wire.NETWORK);
  StringCodec.encode(method, writer);
  StringCodec.encode(url, writer);
  writer.writeUint16(status);
  writer.writeUint32(durationMs);
  VarInt.encode(timestamp, writer);
  return writer.toBytes();
}

// Encode a URL update datagram
export function encodeUrlUpdateDatagram(url, title) {
  const writer = new ByteWriter();
  writer.writeUint8(Wire.URL_UPDATE);
  StringCodec.encode(url, writer);
  StringCodec.encode(title, writer);
  return writer.toBytes();
}

// Encode a pong datagram
export function encodePongDatagram() {
  const writer = new ByteWriter();
  writer.writeUint8(Wire.PONG);
  return writer.toBytes();
}

// Client type constants
export const ClientType = {
  EXTENSION: 0,
  WEBUI: 1,
};

// Encode a register message
export function encodeRegister(name, userAgent, clientType = ClientType.EXTENSION) {
  const writer = new ByteWriter();
  writer.writeUint8(Wire.REGISTER);
  StringCodec.encode(name, writer);
  StringCodec.encode(userAgent, writer);
  writer.writeUint8(clientType);
  return writer.toBytes();
}

// Encode a command result
export function encodeCommandResult(id, success, result, error) {
  const writer = new ByteWriter();
  writer.writeUint8(Wire.COMMAND_RESULT);
  VarInt.encode(id, writer);
  writer.writeUint8(success ? 1 : 0);
  StringCodec.encode(result || '', writer);
  StringCodec.encode(error || '', writer);
  return writer.toBytes();
}

// Encode screenshot data (for uni stream)
export function encodeScreenshotData(id, data, url, title) {
  const writer = new ByteWriter();
  writer.writeUint8(Wire.SCREENSHOT_DATA);
  VarInt.encode(id, writer);
  BytesCodec.encode(data, writer);
  StringCodec.encode(url, writer);
  StringCodec.encode(title, writer);
  return writer.toBytes();
}

// Decode a control message from stream
export async function decodeControlMessage(reader) {
  const msgType = await reader.readUint8();

  switch (msgType) {
    case Wire.REGISTER_OK:
      return { type: 'register_ok' };

    case Wire.REGISTER_ERROR: {
      const code = await reader.readUint16();
      const message = await StringCodec.decode(reader);
      return { type: 'register_error', code, message };
    }

    default:
      throw new Error(`Unknown control message type: 0x${msgType.toString(16)}`);
  }
}

// Decode a server command from stream
export async function decodeCommand(reader) {
  const msgType = await reader.readUint8();

  switch (msgType) {
    case Wire.PING:
      return { type: 'ping' };

    case Wire.REQUEST_SCREENSHOT: {
      const id = await VarInt.decode(reader);
      return { type: 'screenshot', id: Number(id) };
    }

    case Wire.EXECUTE_JS: {
      const id = await VarInt.decode(reader);
      const code = await StringCodec.decode(reader);
      return { type: 'execute_js', id: Number(id), code };
    }

    case Wire.CLICK: {
      const id = await VarInt.decode(reader);
      const selector = await StringCodec.decode(reader);
      return { type: 'click', id: Number(id), selector };
    }

    case Wire.FILL: {
      const id = await VarInt.decode(reader);
      const selector = await StringCodec.decode(reader);
      const value = await StringCodec.decode(reader);
      return { type: 'fill', id: Number(id), selector, value };
    }

    case Wire.GET_TEXT: {
      const id = await VarInt.decode(reader);
      const selector = await StringCodec.decode(reader);
      return { type: 'get_text', id: Number(id), selector };
    }

    case Wire.NAVIGATE: {
      const id = await VarInt.decode(reader);
      const url = await StringCodec.decode(reader);
      return { type: 'navigate', id: Number(id), url };
    }

    default:
      throw new Error(`Unknown command type: 0x${msgType.toString(16)}`);
  }
}

// Log level mapping
export const LOG_LEVELS = {
  trace: 0,
  debug: 1,
  log: 2,
  info: 2,
  warn: 3,
  error: 4,
  fatal: 5,
};

export function logLevelToNumber(level) {
  return LOG_LEVELS[level] ?? 2;
}
