// Perch Browser Observer - Background Script
// Manages WebTransport connection to Perch server and handles commands

// Chrome <144 compatibility: alias chrome to browser
if (typeof browser === "undefined") {
    globalThis.browser = chrome;
}

import {
    encodeLogDatagram,
    encodeNetworkDatagram,
    encodeUrlUpdateDatagram,
    encodePongDatagram,
    encodeRegister,
    encodeCommandResult,
    encodeScreenshotData,
    decodeControlMessage,
    decodeCommand,
    logLevelToNumber,
    StreamReader,
} from "./codec.js";

console.log("[Perch] Background script loading...");

const DEFAULT_PROFILE = {
    name: "default",
    url: "https://wt.perch.perhaps.works:443/",
    user: "",
    pass: "",
};

const DEFAULT_CONFIG = {
    profiles: [DEFAULT_PROFILE],
    activeProfile: "default",
    extensionName: "firefox-1",
    enabled: true,
    severityFilter: "all",
};

const SEVERITY_LEVELS = {
    all: 0,
    trace: 0,
    debug: 1,
    info: 2,
    warn: 3,
    error: 4,
};

let config = { ...DEFAULT_CONFIG };
let transport = null;
let session = null;
let controlSend = null; // Writer for control stream
let datagramWriter = null; // Reusable datagram writer
let reconnectAttempts = 0;
let reconnectTimeout = null;
let networkRequests = new Map();
let lastError = null;

// Load config from storage
async function loadConfig() {
    try {
        const stored = await browser.storage.local.get([
            "extensionName",
            "enabled",
            "severityFilter",
            "profiles",
            "activeProfile",
        ]);

        // Merge stored profile data with defaults
        const defaultProfile = { ...DEFAULT_PROFILE };
        const storedProfile = stored.profiles?.[0] || {};
        const mergedProfile = {
            ...defaultProfile,
            pinnedTabId: storedProfile.pinnedTabId || null,
            pinnedOrigin: storedProfile.pinnedOrigin || null,
        };

        config = {
            profiles: [mergedProfile],
            activeProfile: stored.activeProfile || "default",
            extensionName: stored.extensionName || DEFAULT_CONFIG.extensionName,
            enabled:
                stored.enabled !== undefined
                    ? stored.enabled
                    : DEFAULT_CONFIG.enabled,
            severityFilter:
                stored.severityFilter || DEFAULT_CONFIG.severityFilter,
        };
    } catch (e) {
        console.error("[Perch] Failed to load config:", e);
    }
}

function getActiveProfile() {
    return (
        config.profiles.find((p) => p.name === config.activeProfile) ||
        config.profiles[0] ||
        DEFAULT_PROFILE
    );
}

async function saveConfig(newConfig) {
    const oldActiveProfile = config.activeProfile;
    config = { ...config, ...newConfig };
    await browser.storage.local.set(config);

    if (
        newConfig.activeProfile &&
        newConfig.activeProfile !== oldActiveProfile
    ) {
        disconnect();
        reconnectAttempts = 0;
        if (config.enabled) {
            connect();
        }
    }

    if (newConfig.enabled !== undefined) {
        if (newConfig.enabled && !isConnected()) {
            connect();
        } else if (!newConfig.enabled && transport) {
            disconnect();
        }
    }
}

async function updateBadge() {
    if (!isConnected()) {
        // Red: disconnected
        browser.action.setBadgeBackgroundColor({ color: "#ef4444" });
        browser.action.setBadgeText({ text: "!" });
        return;
    }

    const pinnedTabId = getPinnedTabId();
    if (!pinnedTabId) {
        // Green: connected, no tab pinned (monitoring all)
        browser.action.setBadgeBackgroundColor({ color: "#22c55e" });
        browser.action.setBadgeText({ text: "" });
        return;
    }

    // Check if active tab is the pinned tab
    try {
        const [activeTab] = await browser.tabs.query({
            active: true,
            currentWindow: true,
        });
        if (activeTab && activeTab.id === pinnedTabId) {
            // Green: connected + viewing pinned tab
            browser.action.setBadgeBackgroundColor({ color: "#22c55e" });
            browser.action.setBadgeText({ text: "" });
        } else {
            // Orange: connected but wrong tab
            browser.action.setBadgeBackgroundColor({ color: "#f97316" });
            browser.action.setBadgeText({ text: "○" });
        }
    } catch {
        browser.action.setBadgeBackgroundColor({ color: "#22c55e" });
        browser.action.setBadgeText({ text: "" });
    }
}

function getReconnectDelay() {
    const delays = [5000, 10000, 20000, 40000, 60000];
    return delays[Math.min(reconnectAttempts, delays.length - 1)];
}

function isConnected() {
    return !!(transport && session);
}

// Build WebTransport URL with auth query param
// Uses wt.{hostname} subdomain on port 443/udp for QUIC passthrough
function buildUrl() {
    const profile = getActiveProfile();
    let url = profile.url;

    // Add wt. prefix to hostname and use port 443 for WebTransport
    try {
        const parsed = new URL(url);
        if (!parsed.hostname.startsWith("wt.")) {
            parsed.hostname = `wt.${parsed.hostname}`;
        }
        parsed.port = "443";
        url = parsed.toString();
    } catch (e) {
        console.error("[Perch] Failed to parse URL:", e);
    }

    // Add auth as query param if credentials provided
    if (profile.user && profile.pass) {
        const auth = btoa(`${profile.user}:${profile.pass}`);
        const sep = url.includes("?") ? "&" : "?";
        url = `${url}${sep}auth=${encodeURIComponent(auth)}`;
    }

    return url;
}

// Connect to Perch via WebTransport
async function connect() {
    if (!config.enabled) return;
    if (transport) return;

    const url = buildUrl();
    console.log("[Perch] Connecting to", url);

    try {
        transport = new WebTransport(url);

        await transport.ready;
        console.log("[Perch] WebTransport ready");

        // Get session (WebTransport gives us the connection directly)
        session = transport;

        // Open control bidi stream for registration
        const controlStream = await session.createBidirectionalStream();
        controlSend = controlStream.writable.getWriter();
        const controlRecv = controlStream.readable.getReader();

        // Send registration
        const registerMsg = encodeRegister(
            config.extensionName,
            navigator.userAgent,
        );
        await controlSend.write(registerMsg);

        // Read registration response
        const reader = new StreamReader({ read: () => controlRecv.read() });
        const response = await decodeControlMessage(reader);

        if (response.type === "register_error") {
            lastError = response.message;
            console.error("[Perch] Registration failed:", response.message);
            updateBadge();
            transport.close();
            transport = null;
            session = null;
            scheduleReconnect();
            return;
        }

        console.log("[Perch] Registered successfully");
        reconnectAttempts = 0;
        lastError = null;
        updateBadge();

        // Set up datagram writer
        datagramWriter = session.datagrams.writable.getWriter();

        // Send current URL
        sendCurrentUrl();

        // Start listening for incoming bidi streams (commands from server)
        listenForCommands();

        // Handle transport close
        transport.closed
            .then(() => {
                console.log("[Perch] Transport closed");
                handleDisconnect();
            })
            .catch((e) => {
                console.error("[Perch] Transport error:", e);
                lastError = e.message || "Connection error";
                handleDisconnect();
            });
    } catch (e) {
        // WebTransportError has special properties
        const errorMsg = e.message || e.toString();
        const errorSource = e.source || "";
        const errorCode = e.streamErrorCode ?? e.sessionErrorCode ?? "";
        console.error(
            "[Perch] Connection failed:",
            errorMsg,
            errorSource ? `(source: ${errorSource})` : "",
            errorCode ? `(code: ${errorCode})` : "",
        );
        lastError = errorMsg || "Connection failed";
        updateBadge();
        transport = null;
        session = null;
        scheduleReconnect();
    }
}

function handleDisconnect() {
    transport = null;
    session = null;
    controlSend = null;
    datagramWriter = null;
    updateBadge();
    if (config.enabled) {
        scheduleReconnect();
    }
}

function disconnect() {
    if (reconnectTimeout) {
        clearTimeout(reconnectTimeout);
        reconnectTimeout = null;
    }

    if (transport) {
        transport.close();
        transport = null;
        session = null;
        controlSend = null;
        datagramWriter = null;
    }

    updateBadge();
}

function scheduleReconnect() {
    if (reconnectTimeout) return;

    const delay = getReconnectDelay();
    console.log(`[Perch] Reconnecting in ${delay / 1000}s...`);

    reconnectTimeout = setTimeout(() => {
        reconnectTimeout = null;
        reconnectAttempts++;
        connect();
    }, delay);
}

// Send a datagram (for logs, network, url updates)
function sendDatagram(data) {
    if (!datagramWriter) {
        console.log("[Perch] sendDatagram: no datagramWriter yet");
        return;
    }
    datagramWriter.write(data).catch((e) => {
        console.error("[Perch] sendDatagram error:", e);
        // Connection likely broken, trigger reconnect
        if (e.name === "InvalidStateError" || e.message?.includes("closed")) {
            console.log(
                "[Perch] Connection appears broken, triggering reconnect",
            );
            handleDisconnect();
        }
    });
}

// Listen for incoming bidi streams (commands from server)
async function listenForCommands() {
    if (!session) return;

    const reader = session.incomingBidirectionalStreams.getReader();

    try {
        while (true) {
            const { value: stream, done } = await reader.read();
            if (done) break;

            // Handle each command stream
            handleCommandStream(stream);
        }
    } catch (e) {
        console.error("[Perch] Error listening for commands:", e);
    }
}

// Handle a command stream from server
async function handleCommandStream(stream) {
    const recv = stream.readable.getReader();
    const send = stream.writable.getWriter();
    const reader = new StreamReader({ read: () => recv.read() });

    try {
        const command = await decodeCommand(reader);
        await handleCommand(command, send);
    } catch (e) {
        console.error("[Perch] Error handling command:", e);
    } finally {
        try {
            await send.close();
        } catch {}
        try {
            recv.releaseLock();
        } catch {}
    }
}

// Handle a command from server
async function handleCommand(command, sendWriter) {
    switch (command.type) {
        case "ping":
            sendDatagram(encodePongDatagram());
            break;

        case "screenshot":
            await takeScreenshot(command.id);
            break;

        case "execute_js":
            await executeInTab(command.id, command.code, sendWriter);
            break;

        case "click":
            await clickInTab(command.id, command.selector, sendWriter);
            break;

        case "fill":
            await fillInTab(
                command.id,
                command.selector,
                command.value,
                sendWriter,
            );
            break;

        case "get_text":
            await getTextInTab(command.id, command.selector, sendWriter);
            break;

        case "navigate":
            await navigateTab(command.id, command.url, sendWriter);
            break;

        default:
            console.log("[Perch] Unknown command:", command.type);
    }
}

// Send command result on the bidi stream
async function sendCommandResult(writer, id, success, result, error) {
    const data = encodeCommandResult(id, success, result, error);
    await writer.write(data);
}

// Check if log should be sent based on severity filter
function shouldSendLog(level) {
    const filterLevel = SEVERITY_LEVELS[config.severityFilter] || 0;
    const logLevel = SEVERITY_LEVELS[level] || 0;
    return logLevel >= filterLevel;
}

// Check if tab matches pinned profile
function matchesPinnedTab(tabId, url) {
    const profile = getActiveProfile();
    if (!profile.pinnedTabId || !profile.pinnedOrigin) return true;
    if (tabId !== profile.pinnedTabId) return false;
    try {
        const urlOrigin = new URL(url).origin;
        return urlOrigin === profile.pinnedOrigin;
    } catch {
        return false;
    }
}

function getPinnedTabId() {
    const profile = getActiveProfile();
    return profile.pinnedTabId || null;
}

// Navigate pinned tab to URL
async function navigateTab(id, url, sendWriter) {
    const tabId = getPinnedTabId();
    if (!tabId) {
        await sendCommandResult(sendWriter, id, false, null, "No tab pinned");
        return;
    }

    try {
        await browser.tabs.update(tabId, { url });
        await sendCommandResult(sendWriter, id, true, "ok", null);
    } catch (e) {
        await sendCommandResult(sendWriter, id, false, null, e.message);
    }
}

// Execute JavaScript in pinned tab
async function executeInTab(id, code, sendWriter) {
    const tabId = getPinnedTabId();
    if (!tabId) {
        await sendCommandResult(sendWriter, id, false, null, "No tab pinned");
        return;
    }

    try {
        const results = await browser.scripting.executeScript({
            target: { tabId },
            func: (code) => {
                try {
                    const result = eval(code);
                    return {
                        success: true,
                        result: result === undefined ? null : String(result),
                    };
                } catch (e) {
                    return { success: false, error: e.message };
                }
            },
            args: [code],
        });

        const result = results[0]?.result;
        if (result?.success) {
            await sendCommandResult(sendWriter, id, true, result.result, null);
        } else {
            await sendCommandResult(
                sendWriter,
                id,
                false,
                null,
                result?.error || "Unknown error",
            );
        }
    } catch (e) {
        await sendCommandResult(sendWriter, id, false, null, e.message);
    }
}

// Click element in pinned tab
async function clickInTab(id, selector, sendWriter) {
    const tabId = getPinnedTabId();
    if (!tabId) {
        await sendCommandResult(sendWriter, id, false, null, "No tab pinned");
        return;
    }

    try {
        const results = await browser.scripting.executeScript({
            target: { tabId },
            func: (selector) => {
                const el = document.querySelector(selector);
                if (!el)
                    return {
                        success: false,
                        error: "Element not found: " + selector,
                    };
                el.click();
                return { success: true };
            },
            args: [selector],
        });

        const result = results[0]?.result;
        if (result?.success) {
            await sendCommandResult(sendWriter, id, true, null, null);
        } else {
            await sendCommandResult(
                sendWriter,
                id,
                false,
                null,
                result?.error || "Click failed",
            );
        }
    } catch (e) {
        await sendCommandResult(sendWriter, id, false, null, e.message);
    }
}

// Fill input in pinned tab
async function fillInTab(id, selector, value, sendWriter) {
    const tabId = getPinnedTabId();
    if (!tabId) {
        await sendCommandResult(sendWriter, id, false, null, "No tab pinned");
        return;
    }

    try {
        const results = await browser.scripting.executeScript({
            target: { tabId },
            func: (selector, value) => {
                const el = document.querySelector(selector);
                if (!el)
                    return {
                        success: false,
                        error: "Element not found: " + selector,
                    };
                el.value = value;
                el.dispatchEvent(new Event("input", { bubbles: true }));
                el.dispatchEvent(new Event("change", { bubbles: true }));
                return { success: true };
            },
            args: [selector, value],
        });

        const result = results[0]?.result;
        if (result?.success) {
            await sendCommandResult(sendWriter, id, true, null, null);
        } else {
            await sendCommandResult(
                sendWriter,
                id,
                false,
                null,
                result?.error || "Fill failed",
            );
        }
    } catch (e) {
        await sendCommandResult(sendWriter, id, false, null, e.message);
    }
}

// Get text from element in pinned tab
async function getTextInTab(id, selector, sendWriter) {
    const tabId = getPinnedTabId();
    if (!tabId) {
        await sendCommandResult(sendWriter, id, false, null, "No tab pinned");
        return;
    }

    try {
        const results = await browser.scripting.executeScript({
            target: { tabId },
            func: (selector) => {
                const el = document.querySelector(selector);
                if (!el)
                    return {
                        success: false,
                        error: "Element not found: " + selector,
                    };
                return { success: true, result: el.textContent };
            },
            args: [selector],
        });

        const result = results[0]?.result;
        if (result?.success) {
            await sendCommandResult(sendWriter, id, true, result.result, null);
        } else {
            await sendCommandResult(
                sendWriter,
                id,
                false,
                null,
                result?.error || "Get text failed",
            );
        }
    } catch (e) {
        await sendCommandResult(sendWriter, id, false, null, e.message);
    }
}

// Take screenshot and send via uni stream
async function takeScreenshot(id) {
    if (!session) return;

    try {
        const [tab] = await browser.tabs.query({
            active: true,
            currentWindow: true,
        });
        if (!tab) {
            console.error("[Perch] No active tab for screenshot");
            return;
        }

        const dataUrl = await browser.tabs.captureVisibleTab(null, {
            format: "jpeg",
            quality: 80,
        });
        const resizedData = await resizeImage(dataUrl, 1024);
        const base64Data = resizedData.replace(/^data:image\/\w+;base64,/, "");

        // Decode base64 to bytes
        const binaryStr = atob(base64Data);
        const bytes = new Uint8Array(binaryStr.length);
        for (let i = 0; i < binaryStr.length; i++) {
            bytes[i] = binaryStr.charCodeAt(i);
        }

        // Send via uni stream
        const stream = await session.createUnidirectionalStream();
        const writer = stream.getWriter();
        const data = encodeScreenshotData(id, bytes, tab.url, tab.title);
        await writer.write(data);
        await writer.close();
    } catch (e) {
        console.error("[Perch] Screenshot failed:", e);
    }
}

// Resize image
async function resizeImage(dataUrl, maxWidth) {
    return new Promise((resolve, reject) => {
        const img = new Image();
        img.onload = () => {
            let width = img.width;
            let height = img.height;

            if (width > maxWidth) {
                const ratio = maxWidth / width;
                width = maxWidth;
                height = Math.round(height * ratio);
            }

            let canvas;
            if (typeof OffscreenCanvas !== "undefined") {
                canvas = new OffscreenCanvas(width, height);
            } else {
                canvas = document.createElement("canvas");
                canvas.width = width;
                canvas.height = height;
            }

            const ctx = canvas.getContext("2d");
            ctx.drawImage(img, 0, 0, width, height);

            if (canvas.convertToBlob) {
                canvas
                    .convertToBlob({ type: "image/jpeg", quality: 0.8 })
                    .then((blob) => {
                        const reader = new FileReader();
                        reader.onload = () => resolve(reader.result);
                        reader.onerror = reject;
                        reader.readAsDataURL(blob);
                    })
                    .catch(reject);
            } else {
                resolve(canvas.toDataURL("image/jpeg", 0.8));
            }
        };
        img.onerror = reject;
        img.src = dataUrl;
    });
}

// Send current tab URL
async function sendCurrentUrl() {
    try {
        const [tab] = await browser.tabs.query({
            active: true,
            currentWindow: true,
        });
        if (tab) {
            sendDatagram(encodeUrlUpdateDatagram(tab.url, tab.title));
        }
    } catch (e) {
        console.error("[Perch] Failed to get current URL:", e);
    }
}

// Track tab URL changes
browser.tabs.onUpdated.addListener((tabId, changeInfo, tab) => {
    if (changeInfo.url && tab.active) {
        sendDatagram(encodeUrlUpdateDatagram(changeInfo.url, tab.title || ""));
    }
});

// Track active tab changes
browser.tabs.onActivated.addListener(async (activeInfo) => {
    try {
        const tab = await browser.tabs.get(activeInfo.tabId);
        sendDatagram(encodeUrlUpdateDatagram(tab.url, tab.title));
    } catch {}
    // Update badge to reflect whether active tab is pinned
    updateBadge();
});

// Track network requests - start
// Note: Firefox uses documentUrl, Chrome uses initiator
browser.webRequest.onBeforeRequest.addListener(
    (details) => {
        if (!config.enabled) return;
        const originUrl =
            details.documentUrl ??
            details.initiator ??
            details.originUrl ??
            details.url;
        if (!matchesPinnedTab(details.tabId, originUrl)) return;
        networkRequests.set(details.requestId, {
            startTime: Date.now(),
            method: details.method,
            url: details.url,
        });
    },
    { urls: ["<all_urls>"] },
);

// Track network requests - completion
browser.webRequest.onCompleted.addListener(
    (details) => {
        if (!config.enabled) return;

        const request = networkRequests.get(details.requestId);
        networkRequests.delete(details.requestId);

        if (!request) return;

        const duration = Date.now() - request.startTime;
        sendDatagram(
            encodeNetworkDatagram(
                request.method,
                request.url,
                details.statusCode,
                duration,
                Date.now(),
            ),
        );
    },
    { urls: ["<all_urls>"] },
);

// Track network request errors
browser.webRequest.onErrorOccurred.addListener(
    (details) => {
        if (!config.enabled) return;

        const request = networkRequests.get(details.requestId);
        networkRequests.delete(details.requestId);

        if (!request) return;

        const duration = Date.now() - request.startTime;
        sendDatagram(
            encodeNetworkDatagram(
                request.method,
                request.url,
                0,
                duration,
                Date.now(),
            ),
        );
    },
    { urls: ["<all_urls>"] },
);

// Clean up stale network requests
setInterval(() => {
    const fiveMinutesAgo = Date.now() - 5 * 60 * 1000;
    for (const [requestId, request] of networkRequests) {
        if (request.startTime < fiveMinutesAgo) {
            networkRequests.delete(requestId);
        }
    }
}, 60000);

// Handle messages from popup and content scripts
console.log("[Perch] Registering message listener");
browser.runtime.onMessage.addListener((message, sender, sendResponse) => {
    console.log("[Perch] Received message:", message.type);
    // Content script: console logs
    if (message.type === "log") {
        const tabId = sender.tab?.id;
        if (
            config.enabled &&
            shouldSendLog(message.level) &&
            matchesPinnedTab(tabId, message.url)
        ) {
            const level = logLevelToNumber(message.level);
            sendDatagram(
                encodeLogDatagram(
                    level,
                    message.message,
                    message.url,
                    message.timestamp,
                    message.source_file,
                    message.line_number,
                ),
            );
        }
        return false;
    }

    // Popup: get status
    if (message.type === "getStatus") {
        const response = {
            connected: isConnected(),
            connecting: transport && !session,
            error: lastError,
            config: config,
        };
        console.log("[Perch] Sending getStatus response:", response);
        sendResponse(response);
        return true;
    }

    if (message.type === "saveConfig") {
        saveConfig(message.config).then(() => {
            updateBadge();
            sendResponse({ success: true });
        });
        return true;
    }

    if (message.type === "reconnect") {
        disconnect();
        reconnectAttempts = 0;
        if (config.enabled) {
            connect();
        }
        sendResponse({ success: true });
        return true;
    }
});

// Initialize
loadConfig().then(() => {
    if (config.enabled) {
        connect();
    } else {
        updateBadge();
    }
});
